/***************************************************************************

Palm Chess Client
Copyright (C) 1999,2000 David Barr (davidbarr@iname.com)

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or (at
your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

****************************************************************************/

#define BOARDTOP   0   /* where on the screen to draw the board */
#define USERFONTID 129 /* the fontid of the font used for messages */
#define FONTHEIGHT 8   /* the height of the font used for message */

/* Main code for chesscli */

#include <PalmOS.h>
#include <sys_socket.h>

#include "chesscliRsc.h"

Err errno;

WinHandle offImage=NULL;
int squaresize=12; /* the size of the bitmaps in pixels */
char currentline[256]; /* the text we just read from the server */
char currentboard[256]; /* the last currentline that began with <12> */
char initstr[256]; /* initialization commands to send to the server */
int sock = -1; /* the socket number of our connection to the server */
int MvTmWh=-1; /* last clock time that white moved */
int MvTmBl=-1; /* last clock time that black moved */
int BlToMv=-1; /* whether or not it is black's turn to move */
UInt32 MvTmLast=-1; /* last system time that either side moved */
int clientflip=0; /* whether or not the board is flipped on the client */
int serverflip=0; /* whether or not the board is flipped on the server */
/* string used to decode style 12 */
static char piecelabels[] = "-PNBRQKpnbrqk";
#define NSAVELINES 16
char savelines[NSAVELINES][81];
BitmapPtr bitmapP[7*2*2];
char ReceiveBuffer[256];
UInt16 currentFormId = -1;

void
putstr(char *s) {
  int wintop, nlines, i;
  RectangleType r, v;
  FormPtr frm;

  frm = FrmGetActiveForm();
  
  if(frm &&
     frm->formId == currentFormId &&
     frm->formId == BoardForm) {
    nlines = (160-squaresize*8)/FONTHEIGHT;
  } else {
    nlines = NSAVELINES;
  }


  wintop=160-FONTHEIGHT*nlines;
  
  r.topLeft.x = 0;
  r.topLeft.y = wintop;
  r.extent.x = 160;
  r.extent.y = nlines*FONTHEIGHT;

  v.topLeft.x = 0;
  v.topLeft.y = wintop+(nlines-1)*FONTHEIGHT;
  v.extent.x = 160;
  v.extent.y = FONTHEIGHT;
  
  if(frm &&
     frm->formId == currentFormId &&
     (frm->formId == BoardForm ||
      frm->formId == MessagesForm)) {
    WinScrollRectangle(&r, winUp, FONTHEIGHT, &v);
    WinEraseRectangle(&v, 1);
    FntSetFont(USERFONTID);
    WinDrawChars(s, StrLen(s), 0, wintop+(nlines-1)*FONTHEIGHT);
  }
  for(i=0; i<(NSAVELINES-1); i++) {
    StrCopy(savelines[i], savelines[i+1]);
  }
  if(StrLen(s) <= 40) {
    StrCopy(savelines[i], s);
  } else {
    StrNCopy(savelines[i], s, 40);
    savelines[i][80] = '\0';
    putstr(s+40);
  }
}

int
CheckForNetwork( void ) {
  Err err;
  UInt16 ifErrs;
  //    Word settingSize = sizeof( DWord );
  //    DWord TraceNetSettings;

  if ( ( err = SysLibFind( "Net.lib", &AppNetRefnum ) ) ) {
    putstr("Net.lib not found");
    return 1;
  }

  err = NetLibOpen( AppNetRefnum, &ifErrs );
  if ( err == netErrAlreadyOpen ) {
    return 0;
  }

  if ( err || ifErrs ) {
    putstr("NetLibOpen failed");
    NetLibClose( AppNetRefnum, false );
    return 2;
  }

  return 0;
}

struct NetSettings_s {
  char host[80];
  char port[80];
  char user[80];
  char pass[80];
};

typedef struct NetSettings_s NetSettings_t;
NetSettings_t NetSettings;

static DmOpenRef ccDB;

static Err
OpenDatabase(void) {
  UInt16        index = 0;
  MemHandle     RecHandle;
  MemPtr        RecPointer;
  Err           err;

  /* Create database, if it doesn't exist, and save default settings status. */
  if (!(ccDB = DmOpenDatabaseByTypeCreator('Data', 'CHCL', dmModeReadWrite))) {
    if ((err = DmCreateDatabase(0, "ChessClientDB", 'CHCL', 'Data', false)))
      return err;
    ccDB = DmOpenDatabaseByTypeCreator('Data', 'CHCL', dmModeReadWrite);

    RecHandle = DmNewRecord(ccDB, &index, sizeof(NetSettings_t));
    DmWrite(MemHandleLock(RecHandle), 0, &NetSettings, sizeof(NetSettings_t));
    MemHandleUnlock(RecHandle);
    DmReleaseRecord(ccDB, index, true);
  }

  // Load a saved game status.
  RecHandle = DmGetRecord(ccDB, 0);
  RecPointer = MemHandleLock(RecHandle);
  MemMove(&NetSettings, RecPointer, sizeof(NetSettings_t));
  MemHandleUnlock(RecHandle);
  DmReleaseRecord(ccDB, 0, true);

  return 0;
}

/*
 * Save game status information.
 */
static void
SaveStatus() {
  MemPtr p = MemHandleLock(DmGetRecord(ccDB, 0));
  //game.seconds = TimGetSeconds() - GameStartedAt;
  DmWrite(p, 0, &NetSettings, sizeof(NetSettings_t));
  MemPtrUnlock(p);
  DmReleaseRecord(ccDB, 0, true);
}

void
handlePenDown(int x, int y) {
  static int prevr=-1, prevc=-1;
  int r, c;
  char move[32];
  int flip;

  flip = clientflip ^ serverflip;
  c = x/squaresize;
  r = 7-(y-BOARDTOP)/squaresize;
  if(flip) {
    c = 7-c;
    r = 7-r;
  }
  StrPrintF(move, "square = %c%d", "abcdefgh"[c], r+1);
  //putstr(move);
  if(c>=0&&c<=7&&r>=0&&r<=7) {
    if(prevr != -1) {
      StrPrintF(move, "%c%d-%c%d\n", "abcdefgh"[prevc], prevr+1,
		"abcdefgh"[c], r+1);
      write(sock, move, StrLen(move));
      prevr=-1;
      prevc=-1;
    } else {
      prevr = r;
      prevc = c;
    }
  } else {
    prevr=-1;
    prevc=-1;
  }
}

static FieldPtr
SetFieldTextFromHandle(FormPtr frm, UInt16 fieldID, MemHandle txtH) {
  MemHandle oldTxtH;
  FieldPtr fldP;
  UInt16 objIdx;

  objIdx = FrmGetObjectIndex(frm, fieldID);
  fldP = FrmGetObjectPtr(frm, objIdx);
  oldTxtH = FldGetTextHandle(fldP);
  FldSetTextHandle(fldP, txtH);
  FldDrawField(fldP);
  if(oldTxtH) {
    MemHandleFree(oldTxtH);
  }
  return fldP;
}

static FieldPtr
SetFieldTextFromStr(FormPtr frm, UInt16 fieldID, char *strP) {
  MemHandle txtH;
  FieldPtr fldP;

  txtH = MemHandleNew(StrLen(strP) + 1);
  if(!txtH) {
    return NULL;
  }
  StrCopy(MemHandleLock(txtH), strP);
  fldP = SetFieldTextFromHandle(frm, fieldID, txtH);
  MemHandleUnlock(txtH);
  return fldP;
}

static void
updateClocks() {
  char TmWhSt[16], TmBlSt[16];
  UInt32 CurrTm = TimGetSeconds();
  int TmWh = MvTmWh, TmBl = MvTmBl;
  FormPtr frm = FrmGetActiveForm();
  int flip,min, sec, hr;

  flip = clientflip ^ serverflip;
  
  if(!frm ||
     frm->formId != currentFormId ||
     frm->formId != BoardForm) {
    return;
  }
  if(MvTmLast == -1) {
    return;
  }
  if(BlToMv) {
    TmBl -= (CurrTm - MvTmLast);
  } else {
    TmWh -= (CurrTm - MvTmLast);
  }

  sec=TmWh%60;
  min=(TmWh/60)%60;
  hr =(TmWh/(60*60));
  if(hr) {
    StrPrintF(TmWhSt, "%d:%02d", hr, min);
  } else {
    StrPrintF(TmWhSt, "%d:%02d", min, sec);
  }

  sec=TmBl%60;
  min=(TmBl/60)%60;
  hr =(TmBl/(60*60));
  if(hr) {
    StrPrintF(TmBlSt, "%d:%02d", hr, min);
  } else {
    StrPrintF(TmBlSt, "%d:%02d", min, sec);
  }
  WinDrawChars(TmWhSt, StrLen(TmWhSt), 145, flip ? -1 : 7);
  WinDrawChars(TmBlSt, StrLen(TmBlSt), 145, flip ? 7 : -1);
}

void
updateBoard(int newdata) {
  char tmpstr[256];
  char *fields[48], *whitename, *blackname;
  int i, nfields=0, cll, j, k, labellen, row, col;
  int flip;
  RectangleType r;

  FormPtr frm;

  if(newdata) {
    MvTmLast = TimGetSeconds();
    StrCopy(currentboard, currentline);
  }

  frm = FrmGetActiveForm();
  
  if(!frm ||
     frm->formId != currentFormId ||
     frm->formId != BoardForm) {
    return;
  }

  StrCopy(tmpstr, currentline);
  cll = StrLen(tmpstr);
  for(i=0; i<cll; i++) {
    if(tmpstr[i] == ' ') {
      fields[nfields++] = &tmpstr[i+1];
      tmpstr[i] = 0;
    }
  }

  whitename = fields[16];
  blackname = fields[17];
  MvTmWh = atoi(fields[23]);
  MvTmBl = atoi(fields[24]);
  serverflip = atoi(fields[29]);

  flip = clientflip ^ serverflip;

  BlToMv = (fields[8][0] == 'B'); /* see who's turn it is */

/*    SetFieldTextFromStr(frm, UpPlayerLbl, flip ? whitename : blackname); */
/*    SetFieldTextFromStr(frm, DownPlayerLbl, flip ? blackname : whitename); */
  updateClocks();

  labellen = StrLen(piecelabels);
  for(i=0; i<8; i++) {
    row = i;
    if(flip) {
      row = 7-row;
    }
    for(j=0; j<8; j++) {
      col = j;
      if(flip) {
	col = 7-col;
      }
      for(k=0; k<labellen; k++) {
	if(fields[i][j] == piecelabels[k]) {
	  break;
	}
      } /* end for k */
      if((i+j)%2) {
	//bmpno++;
      }
      //saveno = (pieceside*7+piecetype)*2+background;
      r.topLeft.x=0;
      r.topLeft.y=0;
      if(k>=7) {
	r.topLeft.y += (squaresize*2);
	k-=6;
      }
      r.topLeft.y += (((i+j+k+1)%2)*squaresize);
      r.topLeft.x=k*squaresize;
      r.extent.x=squaresize;
      r.extent.y=squaresize;
      WinCopyRectangle(offImage, WinGetActiveWindow(), &r,
		       col*squaresize, row*squaresize+BOARDTOP, winPaint);

      //WinDrawBitmap (bitmapP[k*2+((i+j)%2)], col*SQUARESIZE,
      //row*SQUARESIZE+BOARDTOP);
    } /* end for j */
  } /* end for i */
} /* end updateBoard */

void
loadbitmap(int size, int bmpno) {
  WinHandle tmpHandle, tmpHandle2;
  Err err;
  MemHandle bitmapHandle;
  int oldsize;
  RectangleType r;

  oldsize=squaresize;
  squaresize=size;

  if(offImage) {
    WinDeleteWindow(offImage, false);
  }
  
  tmpHandle2 = WinGetDrawWindow();
  offImage=WinCreateOffscreenWindow(squaresize*7, squaresize*4,
				    screenFormat, &err);
  tmpHandle = WinSetDrawWindow(offImage);
  bitmapHandle = DmGetResource('Tbmp',bmpno);
  WinDrawBitmap(MemHandleLock(bitmapHandle),0,0);
  MemHandleUnlock(bitmapHandle);
  WinSetDrawWindow(tmpHandle2);

  if(StrLen(currentboard)) {
    StrCopy(currentline, currentboard);
  } else {
    StrCopy(currentline,
	    "<12> rnbqkbnr pppppppp -------- -------- -------- -------- "
	    "PPPPPPPP RNBQKBNR W -1 0 0 1 1 0 7 White Black 1 2 12 39 39 "
	    "119 122 2 null (0:06) null 0");
  }

  if(currentFormId == BoardForm) {
    r.topLeft.x=0;
    r.topLeft.y=0;
    r.extent.x=oldsize*8;
    r.extent.y=oldsize*8;
    WinEraseRectangle(&r, 1);
  }
  updateBoard(0);
}

static Boolean
BoardFormHandleEvent (EventPtr e) {
  Boolean handled = false;
  FormPtr frm;
  int wintop, i, lines;
    
  switch (e->eType) {
  case frmOpenEvent:
    frm = FrmGetActiveForm();
    FrmDrawForm(frm);
    currentFormId=frm->formId;
    if(StrLen(currentboard)) {
      StrCopy(currentline, currentboard);
    } else {
      StrCopy(currentline,
	      "<12> rnbqkbnr pppppppp -------- -------- -------- -------- "
	      "PPPPPPPP RNBQKBNR W -1 0 0 1 1 0 7 White Black 1 2 12 39 39 "
	      "119 122 2 null (0:06) null 0");
    }
    updateBoard(0);
    lines = (160-squaresize*8)/FONTHEIGHT;
    wintop=160-FONTHEIGHT*lines;
    FntSetFont(USERFONTID);
    for(i=0; i<lines; i++) {
      WinDrawChars(savelines[NSAVELINES-lines+i],
		   StrLen(savelines[NSAVELINES-lines+i]), 0,
		   wintop+i*FONTHEIGHT);
    }
    handled = true;
    break;

  case menuEvent:
    MenuEraseStatus ((MenuBarPtr) MenuGetActiveMenu());

    switch(e->data.menu.itemID) {
    case HelpMenuAbout:
      FrmPopupForm(AboutForm);
      break;
    case OptMenu12:
      loadbitmap(12, PieceBmp12);
      break;
    case OptMenu16:
      loadbitmap(16, PieceBmp16);
      break;
    case OptMenu18:
      loadbitmap(18, PieceBmp18);
      break;
    }

    handled = true;
    break;

  case ctlSelectEvent:
    switch(e->data.ctlSelect.controlID) {
    case ToMessagesBtn:
      currentFormId = -1;
      FrmGotoForm(MessagesForm);
      break;
    case ToLoginBtn:
      currentFormId = -1;
      FrmGotoForm(LoginForm);
      break;
    }
    break;

  case nilEvent:
    break;

  case penDownEvent:
    handlePenDown(e->screenX, e->screenY);
    break;

  case penUpEvent:
    break;

  case penMoveEvent:
    break;

  case ctlEnterEvent:
    break;

  default:
    //StrPrintF(debugstr, "main form event type = %d", e->eType);
    //putstr(debugstr);  /* 0 1 2 7 9 */
    break;
  }

  return handled;
}

static struct in_addr *
atoaddr(char *address) {
  struct hostent *host;
  static struct in_addr saddr;

  /* First try it as aaa.bbb.ccc.ddd. */
  saddr.s_addr = inet_addr(address);
  if (saddr.s_addr != -1) {
    return &saddr;
  }
  host = gethostbyname(address);
  if (host != NULL) {
    return (struct in_addr *) *host->h_addr_list;
  }
  return NULL;
}

static Boolean
LoginFormHandleEvent (EventPtr e) {
  Boolean handled = false;
  static FormPtr frm;
  char *serverPtr, *portPtr, *userPtr, *passPtr, nullStr[1];
  struct sockaddr_in address;
  struct in_addr *addr;
    
  switch (e->eType) {
  case frmOpenEvent:
    frm = FrmGetActiveForm();
    FrmDrawForm(frm);
    currentFormId = frm->formId;
    SetFieldTextFromStr(frm, ServerFld, NetSettings.host);
    SetFieldTextFromStr(frm, PortFld, NetSettings.port);
    SetFieldTextFromStr(frm, UserFld, NetSettings.user);
    SetFieldTextFromStr(frm, PassFld, NetSettings.pass);
    handled = true;
    break;

  case menuEvent:
    MenuEraseStatus(NULL);

    switch(e->data.menu.itemID) {
    }

    handled = true;
    break;

  case ctlSelectEvent:
    switch(e->data.ctlSelect.controlID) {
    case ConnectBtn:
      StrCopy(nullStr, "");
      serverPtr = FldGetTextPtr(FrmGetObjectPtr(frm, FrmGetObjectIndex(frm, ServerFld)));
      portPtr   = FldGetTextPtr(FrmGetObjectPtr(frm, FrmGetObjectIndex(frm, PortFld)));
      userPtr   = FldGetTextPtr(FrmGetObjectPtr(frm, FrmGetObjectIndex(frm, UserFld)));
      passPtr   = FldGetTextPtr(FrmGetObjectPtr(frm, FrmGetObjectIndex(frm, PassFld)));
      if(!serverPtr) { serverPtr = nullStr; }
      if(!portPtr)   { portPtr = nullStr; }
      if(!userPtr)   { userPtr = nullStr; }
      if(!passPtr)   { passPtr = nullStr; }
      StrCopy(NetSettings.host, serverPtr);
      StrCopy(NetSettings.port, portPtr);
      StrCopy(NetSettings.user, userPtr);
      StrCopy(NetSettings.pass, passPtr);
      
      sock = socket(AF_INET, SOCK_STREAM, 0);
      memset((char *) &address, 0, sizeof(address));
      addr = atoaddr(serverPtr);
      address.sin_family = AF_INET;
      address.sin_port = atoi(portPtr);
      address.sin_addr.s_addr = addr->s_addr;
      connect(sock, (struct sockaddr *) &address, sizeof(address));
      StrCopy(currentline, "");
      StrPrintF(initstr, "%s\n%s\nset style 12\n", userPtr, passPtr);
      currentFormId = -1;
      FrmGotoForm(BoardForm);
    }
    break;

  case nilEvent:
    break;

  case penDownEvent:
    break;

  case penUpEvent:
    break;

  case ctlEnterEvent:
    break;

  case keyDownEvent:
    break;

  default:
    break;
  }

  return handled;
}

static Boolean
MessagesFormHandleEvent (EventPtr e) {
  Boolean handled = false;
  FormPtr frm;
  int wintop, i;
    
  switch (e->eType) {
  case frmOpenEvent:
    frm = FrmGetActiveForm();
    FrmDrawForm(frm);
    currentFormId=frm->formId;
    wintop=160-FONTHEIGHT*NSAVELINES;
    FntSetFont(USERFONTID);
    for(i=0; i<NSAVELINES; i++) {
      WinDrawChars(savelines[i], StrLen(savelines[i]), 0,
		   wintop+(i)*FONTHEIGHT);
    }
    handled = true;
    break;

  case menuEvent:
    MenuEraseStatus(NULL);

    switch(e->data.menu.itemID) {
    }

    handled = true;
    break;

  case ctlSelectEvent:
    switch(e->data.ctlSelect.controlID) {
    case ToBoardBtn:
      currentFormId = -1;
      FrmGotoForm(BoardForm);
    }
    break;

  case nilEvent:
    break;

  case penDownEvent:
    break;

  case penUpEvent:
    break;

  case ctlEnterEvent:
    break;

  case keyDownEvent:
    if (e->data.keyDown.chr == '\n') {
      UInt16 focus;
      FormPtr frm;
      FieldPtr fld;
	
      frm = FrmGetActiveForm ();
      focus = FrmGetFocus (frm);
      fld = FrmGetObjectPtr(frm, focus);
	
      StrPrintF(currentline, "%s\n", FldGetTextPtr(fld));
	 
      write(sock, currentline, StrLen(currentline));
      handled = true;
    }
    break;

  default:
    //StrPrintF(debugstr, "main form event type = %d", e->eType);
    //putstr(debugstr);  /* 0 1 2 7 9 */
    break;
  }

  return handled;
}

static Boolean
AboutFormHandleEvent (EventPtr e) {
  Boolean handled = false;
  FormPtr frm;
  //  int wintop, i;
    
  switch (e->eType) {
  case frmOpenEvent:
    frm = FrmGetActiveForm();
    FrmDrawForm(frm);
    currentFormId = frm->formId;
    handled = true;
    break;
  case ctlSelectEvent:
    if (e->data.ctlSelect.controlID == FormAboutButton) {
      FrmReturnToForm(0);
      handled = true; 
      break;
    }
    break;
  default:
    break;
  }

  return handled;
}

void
mydebug(int lineno) {
  // do nothing
}

static Boolean
ApplicationHandleLine() {
  int i;
  
  //StrCopy(debugstr, currentline);
  //mydebug(__LINE__);
  for(i=0; i<5; i++) {
    if("<12> "[i] != currentline[i]) {
      break;
    }
  }
  if(i==5) {
    updateBoard(1);
  } else {
    putstr(currentline);
  }
  return true;
}

static Boolean
ApplicationHandleData() {
  Err i;
  int cll, j;
  static int sentinitstr=0;

  if(!sentinitstr) {
    write(sock, initstr, StrLen(initstr));
    sentinitstr=1;
  }

  //printf("trying ReceiveData...\n");
  cll = StrLen(currentline);
  if(cll == 255) {
    StrCopy(currentline, "");
    cll = 0;
  }
  i = read(sock, ReceiveBuffer, sizeof(ReceiveBuffer));
  if(!i) {
    //StrCopy(debugstr, "remote side closed connection\n");
    //mydebug(__LINE__);
    close(sock);
    sock = -1;
  } else if(i == -1) {
    StrPrintF(ReceiveBuffer, "NetLibReceive error = 0x%04x", errno);
    putstr(ReceiveBuffer);
    i=0;
  }
  for(j=0; j<i; j++) {
    if(ReceiveBuffer[j] == '\n' || ReceiveBuffer[j] == '\r') {
      if(cll) {
	ApplicationHandleLine();
	StrCopy(currentline, "");
	cll = 0;
      }
      continue;
    }
    currentline[cll++] = ReceiveBuffer[j];
    currentline[cll] = 0;
  }
  return true;
}
  

static Boolean
ApplicationHandleEvent(EventPtr e) {
  FormPtr frm;
  UInt16    formId;
  Boolean handled = false;

  if (e->eType == frmLoadEvent) {
    formId = e->data.frmLoad.formID;
    frm = FrmInitForm(formId);
    FrmSetActiveForm(frm);
    switch(formId) {
    case BoardForm:
      FrmSetEventHandler(frm, BoardFormHandleEvent);
      break;
    case MessagesForm:
      FrmSetEventHandler(frm, MessagesFormHandleEvent);
      break;
    case LoginForm:
      FrmSetEventHandler(frm, LoginFormHandleEvent);
      break;
    case AboutForm:
      FrmSetEventHandler(frm, AboutFormHandleEvent);
      break;
    }
    handled = true;
	
    //printf("all done\n");
  }

  return handled;
}

/* Get preferences, open (or create) app database */
static UInt16
StartApplication(void) {
  MemHandle fontHandle;
  FontType *fontPtr;
  int i;
  char tmpstr[64];

  loadbitmap(12, PieceBmp12);

  for(i=0; i<NSAVELINES; i++) {
    savelines[i][0] = '\0';
  }

  putstr("trying CheckForNetwork...");
  if((i=CheckForNetwork())) {
    StrPrintF(tmpstr, "checkfornetwork failed (%d)", i);
    putstr(tmpstr);
  }
  
  // 'Font' and 0x3000 are resource type and ID, defined by you in
  // resource description
  fontHandle=DmGetResource('Font',0x3000); // minico 9
  fontPtr=MemHandleLock(fontHandle);
  // user defined fonts start from 129
  FntDefineFont(USERFONTID,fontPtr);

  OpenDatabase();

  currentFormId = -1;
  FrmGotoForm(LoginForm);
  return 0;
}

/* Save preferences, close forms, close app database */
static void
StopApplication(void) {
  MemHandle fontHandle;

  FrmSaveAllForms();
  FrmCloseAllForms();

  // unlock our font handle
  fontHandle=DmGetResource('Font',0x3000);
  MemHandleUnlock(fontHandle);

  // unlock the piece bitmaps
  if(offImage) {
    WinDeleteWindow(offImage, false);
  }

  if(sock != -1) {
    close(sock);
  }

  NetLibClose(AppNetRefnum, false);
  SaveStatus();
}

/* The main event loop */
static void
EventLoop(void) {
  UInt16 err;
  EventType e;
  NetFDSetType readFDs,writeFDs,exceptFDs, setsize=1;

  StrCopy(currentboard, "");
  
  for(;;) {
    netFDZero(&readFDs);
    setsize = 1;
    netFDSet(sysFileDescStdIn, &readFDs);
    setsize = sysFileDescStdIn+1;
    if(sock != (UInt16)-1) {
      netFDSet(sock, &readFDs);
      if(sock >= setsize) {
	setsize = sock+1;
      }
      NetLibSelect(AppNetRefnum, setsize, &readFDs, &writeFDs, &exceptFDs,
		   sysTicksPerSecond, &err);
      if(netFDIsSet(sock, &readFDs)) {
	ApplicationHandleData();
      }
      /*        if(!netFDIsSet(sysFileDescStdIn, &readFDs)) { */
      /*  	continue; */
      /*        } */
    }

    //EvtGetEvent(&e, evtWaitForever);
    EvtGetEvent(&e, 0);
    
    //printf("event = %d\n", e.eType);
    if (SysHandleEvent (&e)) {
      continue;
    }
    if (MenuHandleEvent (NULL, &e, &err)) {
      continue;
    }
    if (ApplicationHandleEvent (&e)) {
      continue;
    }
    
    FrmDispatchEvent (&e);
    
    if(e.eType == appStopEvent) {
      return;
    }
    
    updateClocks();
  }
}

UInt32
PilotMain(UInt16 cmd, void *cmdPBP, UInt16 launchFlags) {
  UInt16 err;

  if (cmd == sysAppLaunchCmdNormalLaunch) {
    err = StartApplication();
    if (err) {
      return err;
    }

    EventLoop();
    StopApplication();
  } else {
    return sysErrParamErr;
  }

  return 0;
}
