/* $Id: datebook.h,v 1.22 2003/07/16 04:26:13 chrisf Exp $ */

/*
Hot Date - A DatebookDB displayer for the PalmPilot
Copyright (C) 1999 Chris Faherty

Portions copyright (C) 1998 3Com Corporation or its subsidiaries.
All rights reserved.

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
*/

#ifndef __DATEBOOK_H__
#define __DATEBOOK_H__

typedef struct {
    UInt renamedCategories; /* bitfield of categories with a different name */
    char categoryLabels[dmRecNumCategories][dmCategoryLength];
    Byte categoryUniqIDs[dmRecNumCategories];
    Byte lastUniqID;        /* Uniq IDs generated by the device are between */
                            /* 0 - 127.  Those from the PC are 128 - 255.   */
    Byte reserved1;         /* from the compiler word aligning things */
    Word reserved2;
    Byte startOfWeek;
} ApptAppInfoType;

typedef ApptAppInfoType * ApptAppInfoPtr;

/************************************************************
 *
 * Appointment Database constants.
 *
 *************************************************************/
#define apptMaxPerDay  100      /* max appointments displayable on a day. */
#define apptNoTime     -1       /* start time of an untimed appt. */
#define apptNoEndDate  -1       /* end date of appts that repeat forever */
#define apptNoAlarm    -1
#define apptMaxEndTime 0x1737   /* 11:55 pm, hours in high byte, minutes */
                                /* in low byte                           */

#define apptMaxDisplayableAlarms 10

/*
 * Alarm advance - the period of time before the appointment that the 
 * alarm should sound.
 */
typedef enum alarmTypes {aauMinutes, aauHours, aauDays} AlarmUnitType;

typedef struct {
    SByte advance;                /* Alarm advance (-1 = no alarm) */
    AlarmUnitType advanceUnit;    /* minutes, hours, days */
} AlarmInfoType;

/*
 * The following enum is used to specify the frequency of
 * repeating appointments.
 */
enum repeatTypes {
    repeatNone,
    repeatDaily,
    repeatWeekly,
    repeatMonthlyByDay,
    repeatMonthlyByDate,
    repeatYearly
};
typedef enum repeatTypes RepeatType;

/*
 * This structure contains information about repeat appointments.  The 
 * repeatOn member is only used by weelky and monthly-by-day repeating
 * appointments.  For weekly the byte is a bit field that contains the 
 * days of the week the appointments occurs on (bit: 0-sun, 1-mon, 
 * 2-tue, etc.).  For monthly-by-day the byte contains the day the 
 * appointments occurs, (ex: the 3rd friday), the byte is of type 
 * DayOfMonthType.
 */
typedef struct {
    RepeatType repeatType;           /* daily, weekly, monthlyByDay, etc. */
    DateType repeatEndDate;          /* minus one if forever */
    unsigned char repeatFrequency;   /* i.e. every 2 days if repeatType daily */
    unsigned char repeatOn;          /* monthlyByDay and repeatWeekly only */
    unsigned char repeatStartOfWeek; /* repeatWeekly only */
} RepeatInfoType;

typedef RepeatInfoType * RepeatInfoPtr;

/*
 * This structure contains information about the date and time of an 
 * appointment.
 */
typedef struct {
    TimeType startTime;      /* Time the appointment starts */
    TimeType endTime;        /* Time the appointment ends */
    DateType date;           /* date of appointment */
} ApptDateTimeType;

/*
 * This is the structure for a repeating appointment's exceptions list.  The
 * exceptions list is a variable length list of dates that the repeating appointment
 * should not appear on.  The list is in chronological order.
 */
typedef struct {
    UInt numExceptions;
    DateType exception;
} ExceptionsListType;

typedef ExceptionsListType * ExceptionsListPtr;

/*
 * This structure describes what information is present in an
 * AppointmentPackedDBRecordType
 */
typedef struct {
    unsigned when :1;         /* set if when info changed (ApptChangeRecord) */
    unsigned alarm :1;        /* set if record contains alarm info */
    unsigned repeat :1;       /* set if record contains repeat info */
    unsigned note :1;         /* set if record contains a note */
    unsigned exceptions :1;   /* set if record contains exceptions list */
    unsigned description :1;            
} ApptDBRecordFlags;

/*
 * ApptDBRecordType
 *
 * This is the record used by the application.  All pointers are either NULL 
 * or point to data within the PackedDB record.  All strings are null 
 * character terminated.
 */
typedef struct {
    ApptDateTimeType * when;
    AlarmInfoType * alarm;
    RepeatInfoType * repeat;
    ExceptionsListType * exceptions;
    CharPtr description;
    CharPtr note;
} ApptDBRecordType;

typedef ApptDBRecordType * ApptDBRecordPtr;

/*
 * ApptGetAppointments returns an array of the following structures.  But
 * see the note below because I pack this into a smaller version after it
 * is sorted.
 */
typedef struct {
    TimeType startTime;
    TimeType endTime;
    UInt recordNum;
    DateType date;           /* I added this, cff */
    Word type;               /* I added this, cff */
} ApptInfoType;
typedef ApptInfoType * ApptInfoPtr;

/*
 * This is the permanent version of the array.  The above structure was only
 * needed to sort it properly.  After that, I can shrink it down because the
 * table draw routines will grab the information directly from the record.
 */
typedef struct {
    UInt recordNum;
    TimeType startTime;
    DateType date;
} LineItemType;
typedef LineItemType * LineItemPtr;

/***********************************************************************
 *
 *    Internal Structures
 *
 ***********************************************************************/

/*
 * The following structure doesn't really exist.  The first field
 * varies depending on the data present.  However, it is convenient
 * (and less error prone) to use when accessing the other information.
 */
typedef struct {
    ApptDateTimeType when;
    ApptDBRecordFlags flags;    /* A flag set for each  datum present */
    char firstField;
} ApptPackedDBRecordType;

typedef ApptPackedDBRecordType * ApptPackedDBRecordPtr;

/*
 * Function prototypes
 */
Word GetWeekNumber (Word month, Word day, Word year, Word weekStartDay);
Int DateCompare(DateType d1, DateType d2);
Int TimeCompare(TimeType t1, TimeType t2);
Boolean ApptRepeatsOnDate(ApptDBRecordPtr apptRec, DateType date);
Boolean ApptFindFirst(DmOpenRef dbP, DateType date, UIntPtr indexP);
Boolean ApptNextRepeat(ApptDBRecordPtr apptRec, DatePtr dateP);
Boolean AddAppointmentToList(VoidHand * apptListH, ULong shrunk, UInt count,
    UInt origcount, TimeType startTime, TimeType endTime, DateType date,
    UInt recordNum, Word type);
UInt ApptGetAppointments(DmOpenRef dbP, DateType date, Word days,
    VoidHand apptLists [], UInt counts []);
ULong ApptGetAlarmTime(ApptDBRecordPtr apptRec, ULong currentTime,
    DateTimeType *actualdt);
UInt ApptGetUntimedAlarms(DmOpenRef dbP, DateType date, Word days,
    VoidHand apptLists [], UInt counts []);

#endif __DATEBOOK_H__
