/******************************************************************************
 *
 * Copyright (c) 1999-2004 PalmSource, Inc. All rights reserved.
 *
 * File: AppMain.c
 *
 *****************************************************************************/

#include <PalmOS.h>
#include <PalmCompatibility.h>
#include <WristPDA.h>
#include <Form.h>
#include "digitalwResources.h"
#include "digitalw.h"

/***********************************************************************
 *
 *	Entry Points
 *
 ***********************************************************************/


/***********************************************************************
 *
 *	Internal Constants
 *
 ***********************************************************************/
#define appFileCreator			'Foss'	// register your own at http://www.palmos.com/dev/creatorid/
#define appVersionNum			0x02
#define appPrefID				0x00
#define appPrefVersionNum		0x02


/***********************************************************************
 *
 *	Internal Functions
 *
 ***********************************************************************/

UInt ApptGetAppointments(DmOpenRef dbP, DateType date, Word days,
    VoidHand apptLists [], UInt counts []);
UInt ApptGetUntimedAlarms(DmOpenRef dbP, DateType date, Word days,
    VoidHand apptLists [], UInt counts []);

typedef struct {
    UInt recordNum;
    TimeType startTime;
    DateType date;
} LineItemType;
typedef LineItemType * LineItemPtr;

typedef struct {
    TimeType startTime;      /* Time the appointment starts */
    TimeType endTime;        /* Time the appointment ends */
    DateType date;           /* date of appointment */
} ApptDateTimeType;

typedef struct {
    unsigned when :1;         /* set if when info changed (ApptChangeRecord) */
    unsigned alarm :1;        /* set if record contains alarm info */
    unsigned repeat :1;       /* set if record contains repeat info */
    unsigned note :1;         /* set if record contains a note */
    unsigned exceptions :1;   /* set if record contains exceptions list */
    unsigned description :1;            
} ApptDBRecordFlags;

typedef struct {
    ApptDateTimeType when;
    ApptDBRecordFlags flags;    /* A flag set for each  datum present */
    char firstField;
} ApptPackedDBRecordType;

typedef ApptPackedDBRecordType * ApptPackedDBRecordPtr;

enum repeatTypes {
    repeatNone,
    repeatDaily,
    repeatWeekly,
    repeatMonthlyByDay,
    repeatMonthlyByDate,
    repeatYearly
};
typedef enum repeatTypes RepeatType;

/*
 * This structure contains information about repeat appointments.  The 
 * repeatOn member is only used by weelky and monthly-by-day repeating
 * appointments.  For weekly the byte is a bit field that contains the 
 * days of the week the appointments occurs on (bit: 0-sun, 1-mon, 
 * 2-tue, etc.).  For monthly-by-day the byte contains the day the 
 * appointments occurs, (ex: the 3rd friday), the byte is of type 
 * DayOfMonthType.
 */
typedef struct {
    RepeatType repeatType;           /* daily, weekly, monthlyByDay, etc. */
    DateType repeatEndDate;          /* minus one if forever */
    unsigned char repeatFrequency;   /* i.e. every 2 days if repeatType daily */
    unsigned char repeatOn;          /* monthlyByDay and repeatWeekly only */
    unsigned char repeatStartOfWeek; /* repeatWeekly only */
} RepeatInfoType;

typedef RepeatInfoType * RepeatInfoPtr;

typedef struct {
    UInt numExceptions;
    DateType exception;
} ExceptionsListType;

typedef ExceptionsListType * ExceptionsListPtr;

/***********************************************************************
 *
 * FUNCTION:    MainFormDoCommand
 *
 * DESCRIPTION: This routine performs the menu command specified.
 *
 * PARAMETERS:  command  - menu item id
 *
 * RETURNED:    nothing
 *
 * REVISION HISTORY:
 *
 *
 ***********************************************************************/
static Boolean MainFormDoCommand(UInt16 command)
{
	Boolean handled = false;
	FormType * pForm;

	switch (command) {
		case MainOptionsAboutStarterApp:
			pForm = FrmInitForm(AboutForm);
			FrmDoDialog(pForm);					// Display the About Box.
			FrmDeleteForm(pForm);
			handled = true;
			break;

	}
	
	return handled;
}


/***********************************************************************
 *
 * FUNCTION:    MainFormHandleEvent
 *
 * DESCRIPTION: This routine is the event handler for the 
 *              "MainForm" of this application.
 *
 * PARAMETERS:  pEvent  - a pointer to an EventType structure
 *
 * RETURNED:    true if the event has handle and should not be passed
 *              to a higher level handler.
 *
 * REVISION HISTORY:
 *
 *
 ***********************************************************************/
static Boolean MainFormHandleEvent(EventType* pEvent)
{
	Boolean 	handled = false;
	FormType* 	pForm;

	switch (pEvent->eType) {
		case menuEvent:
			return MainFormDoCommand(pEvent->data.menu.itemID);

		case frmOpenEvent:
			pForm = FrmGetActiveForm();
			FrmDrawForm(pForm);
			handled = true;
			break;
			
		default:
			break;
	}
	
	return handled;
}


/***********************************************************************
 *
 * FUNCTION:    AppHandleEvent
 *
 * DESCRIPTION: This routine loads form resources and set the event
 *              handler for the form loaded.
 *
 * PARAMETERS:  event  - a pointer to an EventType structure
 *
 * RETURNED:    true if the event has handle and should not be passed
 *              to a higher level handler.
 *
 * REVISION HISTORY:
 *
 *
 ***********************************************************************/
static Boolean AppHandleEvent(EventType* pEvent)
{
	UInt16 		formId;
	FormType* 	pForm;
	Boolean		handled = false;

	if (pEvent->eType == frmLoadEvent) {
		// Load the form resource.
		formId = pEvent->data.frmLoad.formID;
		
		pForm = FrmInitForm(formId);
		FrmSetActiveForm(pForm);

		// Set the event handler for the form.  The handler of the currently
		// active form is called by FrmHandleEvent each time is receives an
		// event.
		switch (formId) {
			case MainForm:
				FrmSetEventHandler(pForm, MainFormHandleEvent);
				break;

			default:
				break;
		}
		handled = true;
	}
	
	return handled;
}


/***********************************************************************
 *
 * FUNCTION:     AppStart
 *
 * DESCRIPTION:  Get the current application's preferences.
 *
 * PARAMETERS:   nothing
 *
 * RETURNED:     Err value errNone if nothing went wrong
 *
 * REVISION HISTORY:
 *
 *
 ***********************************************************************/
static Err AppStart(void)
{
	FrmGotoForm(MainForm);
	return errNone;
}


/***********************************************************************
 *
 * FUNCTION:    AppStop
 *
 * DESCRIPTION: Save the current state of the application.
 *
 * PARAMETERS:  nothing
 *
 * RETURNED:    nothing
 *
 * REVISION HISTORY:
 *
 *
 ***********************************************************************/
static void AppStop(void)
{
	// Close all the open forms.
	FrmCloseAllForms();
}


/***********************************************************************
 *
 * FUNCTION:    AppEventLoop
 *
 * DESCRIPTION: This routine is the event loop for the application.  
 *
 * PARAMETERS:  nothing
 *
 * RETURNED:    nothing
 *
 * REVISION HISTORY:
 *
 *
 ***********************************************************************/
static void AppEventLoop(void)
{
	Err			error;
	EventType	event;

	do {
		EvtGetEvent(&event, evtWaitForever);

		if (SysHandleEvent(&event))
			continue;
			
		if (MenuHandleEvent(0, &event, &error))
			continue;
			
		if (AppHandleEvent(&event))
			continue;

		FrmDispatchEvent(&event);

	} while (event.eType != appStopEvent);
}

void draw_alarm()
{
DmOpenRef	refWatch;
LocalID		dbID;
MemHandle	hRes;
UInt16		iRes;
BitmapPtr       pBmp;

		dbID = DmFindDatabase( 0 , "digitalw" );
		refWatch = DmOpenDatabase( 0, dbID, dmModeReadOnly );
		iRes = DmFindResource( refWatch, 'Tbmp', 3011, NULL );
		hRes = DmGetResourceIndex( refWatch, iRes );
		pBmp = MemHandleLock( hRes );
		WinDrawBitmap( pBmp, 68, 0 );
		MemHandleUnlock( hRes );
}

// Get the current alarm set in the datebook preferences
UInt32 getCurrentAlarm()
{
	UInt16	prefsSize;
	Int16	prefsVersion = noPreferenceFound;
    DatebookPreferenceType prefsP;	

    prefsSize = sizeof (DatebookPreferenceType);

	// First read the state information from the Wrist PDA private preferences.
	prefsVersion = PrefGetAppPreferences (WPdaCreator, 'DB', &prefsP, &prefsSize, true);
	if (prefsVersion > datebookPrefsVersionNum) {
		prefsVersion = noPreferenceFound;
	}
	// If reading the private prefs failed then read the standard preferences.
	if (prefsVersion == noPreferenceFound) {
		prefsVersion = PrefGetAppPreferences (sysFileCDatebook, datebookPrefID, &prefsP, &prefsSize, true);
	}

   // If no prefs found, return default alarm
   if (prefsVersion == noPreferenceFound ) {
   	  return 999; // default soundid
   }
   else {
    return prefsP.alarmSoundUniqueRecID;
   }
}

void PlayAlarmSound(UInt32 uniqueRecID)
{
	Err			err;
	MemHandle	midiH;							// handle of MIDI record
	SndMidiRecHdrType*	midiHdrP;			// pointer to MIDI record header
	UInt8*		midiStreamP;					// pointer to MIDI stream beginning with the 'MThd'
														// SMF header chunk
	UInt16		cardNo;							// card number of System MIDI database
	LocalID		dbID;								// Local ID of System MIDI database
	DmOpenRef	dbP = NULL;						// reference to open database
	UInt16		recIndex;						// record index of the MIDI record to play
	SndSmfOptionsType	smfOpt;					// SMF play options
	DmSearchStateType	searchState;			// search state for finding the System MIDI database
	Boolean		bError = false;				// set to true if we couldn't find the MIDI record
	Int32       testId;
	
	testId = uniqueRecID;
	
	// Find the system MIDI database
	err = DmGetNextDatabaseByTypeCreator(true, &searchState,
			 		sysFileTMidi, sysFileCSystem, true, 
			 		&cardNo, &dbID);
	if ( err )
		bError = true;														// DB not found
	
	// Open the MIDI database in read-only mode
	if ( !bError )
		{
		dbP = DmOpenDatabase (cardNo, dbID, dmModeReadOnly);
		if ( !dbP )
			bError = true;													// couldn't open
		}
	
	// Find the MIDI track record
	if ( !bError )
		{
		err = DmFindRecordByID (dbP, uniqueRecID, &recIndex);
		if ( err )
			bError = true;														// record not found
		}
		
	// Lock the record and play the sound
	if ( !bError )
		{
		// Find the record handle and lock the record
		midiH = DmQueryRecord(dbP, recIndex);
		midiHdrP = MemHandleLock(midiH);
		
		// Get a pointer to the SMF stream
		midiStreamP = (UInt8*)midiHdrP + midiHdrP->bDataOffset;
		
		// Play the sound (ignore the error code)
		// The sound can be interrupted by a key/digitizer event
		smfOpt.dwStartMilliSec = 0;
		smfOpt.dwEndMilliSec = sndSmfPlayAllMilliSec;
		smfOpt.amplitude = PrefGetPreference(prefAlarmSoundVolume);
		smfOpt.interruptible = false;
		smfOpt.reserved = 0;
		err = SndPlaySmf (NULL, sndSmfCmdPlay, midiStreamP, &smfOpt, NULL, NULL, false);
		
		// Unlock the record
		MemPtrUnlock (midiHdrP);
		}
	
	// Close the MIDI database
	if ( dbP )
		DmCloseDatabase (dbP);
	
}

static void play_alarm(void)
{
  PlayAlarmSound( getCurrentAlarm() );
}

void handleAlarm()
{
Err		AlarmTriggered;
UInt32	Ver;

AlarmTriggered = kFossilSystemStatusAlarmNotTriggered;

if ( WPdaGetVersion( & Ver ) == errNone )
	AlarmTriggered = FossilGetSystemStatus( kFossilGetAlarmState );

if ( AlarmTriggered == kFossilSystemStatusAlarmTriggered ) {
    draw_alarm();
    play_alarm();
		}
	
}

static void draw_battery(void)
{
UInt16 batt; 
UInt8 battPercent; 
Coord lbat;
RectangleType rec_bat;
DmOpenRef	refWatch;
LocalID		dbID;
MemHandle	hRes;
UInt16		iRes;
BitmapPtr       pBmp;
const CustomPatternType dark_gray = { 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF };

		// Load the battery resource and draw it
		dbID = DmFindDatabase( 0 , "digitalw" );
		refWatch = DmOpenDatabase( 0, dbID, dmModeReadOnly );
		iRes = DmFindResource( refWatch, 'Tbmp', 5001, NULL );
		hRes = DmGetResourceIndex( refWatch, iRes );
		pBmp = MemHandleLock( hRes );
		WinDrawBitmap( pBmp, 122, 3 );
		MemHandleUnlock( hRes );

		batt = SysBatteryInfo(false, NULL, NULL, NULL, NULL,NULL, &battPercent); 
		lbat = (26 * battPercent) / 100;
		
		// Set the coordinates to 
		rec_bat.topLeft.x = 127;
		rec_bat.topLeft.y = 6;
		rec_bat.extent.x = lbat + 1;
		rec_bat.extent.y = 8;

		// Draw the 'remaining' battery rectangle
		WinSetPattern(&dark_gray);
		WinFillRectangle(&rec_bat,1);
		DmCloseDatabase( refWatch );

}

static void draw_watch(void)
{
DmOpenRef	refWatch;
MemHandle	hRes;
UInt16		iRes;
BitmapPtr       pBmp;
UInt16		hd1, hd2, md1, md2;
DateTimeType Now;
LocalID dbID;
char Str[32];
//
VoidHand apptLists[31];
UInt apptCounts[31];
UInt apptTotal, apptDays;
DmOpenRef DatebookDB;
UInt mode=0;
short i,untimed, total, offset, j, totalapt;
LineItemPtr li;
Char AppErrStr[30];
VoidHand recHandle;
ApptPackedDBRecordPtr apptrecP;
long l1;
DateType date;
int display; // Number of appts displayed

     	WinSetForeColor( UIColorGetTableEntryIndex( UIObjectForeground ) );
	    WinSetBackColor( UIColorGetTableEntryIndex( UIFieldBackground ) );
    	WinSetTextColor( UIColorGetTableEntryIndex( UIFieldText ) );
  		WinEraseWindow();
  		
		dbID = DmFindDatabase( 0 , "digitalw" );
		refWatch = DmOpenDatabase( 0, dbID, dmModeReadOnly );

        
        // Draw the actual time

        // Get the current time
        TimSecondsToDateTime ( TimGetSeconds(), & Now );
        hd1 = Now.hour / 10;
        hd2 = Now.hour % 10;
        md1 = Now.minute / 10;
        md2 = Now.minute % 10;
                       
        // Draw the date
        FntSetFont(FossilLargeFontID(WRISTPDA,boldFont));
        DateTemplateToAscii( "^1l", Now.month, Now.day, Now.year, Str, sizeof( Str ) );
        WinDrawChars( Str, StrLen( Str ), 15, 65);
		DateTemplateToAscii( "^0z-^2r-^4s", Now.month, Now.day, Now.year, Str, sizeof( Str ) );
		WinDrawChars( Str, StrLen( Str ), 90, 65 );

        // First digit hour
		iRes = DmFindResource( refWatch, 'Tbmp', 3000 + hd1, NULL );
		hRes = DmGetResourceIndex( refWatch, iRes );
		pBmp = MemHandleLock( hRes );
		WinDrawBitmap( pBmp, 5, 30 );
		MemHandleUnlock( hRes );

        // Second digit hour
		iRes = DmFindResource( refWatch, 'Tbmp', 3000 + hd2, NULL );
		hRes = DmGetResourceIndex( refWatch, iRes );
		pBmp = MemHandleLock( hRes );
		WinDrawBitmap( pBmp, 40, 30 );
		MemHandleUnlock( hRes );

        // Draw the colon
		iRes = DmFindResource( refWatch, 'Tbmp', 3010, NULL );
		hRes = DmGetResourceIndex( refWatch, iRes );
		pBmp = MemHandleLock( hRes );
		WinDrawBitmap( pBmp, 75, 30 );
		MemHandleUnlock( hRes );

        // First digit minute
		iRes = DmFindResource( refWatch, 'Tbmp', 3000 + md1, NULL );
		hRes = DmGetResourceIndex( refWatch, iRes );
		pBmp = MemHandleLock( hRes );
		WinDrawBitmap( pBmp, 94, 30 );
		MemHandleUnlock( hRes );

        // Second digit minute
		iRes = DmFindResource( refWatch, 'Tbmp', 3000 + md2, NULL );
		hRes = DmGetResourceIndex( refWatch, iRes );
		pBmp = MemHandleLock( hRes );
		WinDrawBitmap( pBmp, 128, 30 );
		MemHandleUnlock( hRes );
   		DmCloseDatabase( refWatch );

    /* This is the new datebook code */		
	/* Open DatebookDB */
	DateSecondsToDate(TimGetSeconds(), &date);
    DatebookDB = DmOpenDatabaseByTypeCreator('DATA', 'date', mode | dmModeReadOnly);

    // set number of days for search
    apptDays = 1;
    untimed = 0; 
    j = 0;
    
    for (i=0; i < 31; i++) {
        apptLists[i] = 0;
        apptCounts[i] = 0;
    }

  /* If the database is opened, do stuff */
  if (DatebookDB) {
            apptTotal = ApptGetAppointments(DatebookDB, date, apptDays,
                apptLists, apptCounts);
//            apptTotal += ApptGetUntimedAlarms(DatebookDB, date, apptDays,
//                apptLists, apptCounts);

// This is the uggliest part of the code
// Some hints:
//   i is the record number to fetch from the datebook db
//   j is the line number used to display the text on the screen
//   total and totalapt are used to ensure that we don't go
//                      above the 5 appts limits
//   display is a counter of the actual number of appts that we
//           have displayed so far
// Go to the first appointment
// apptCounts[0] = today's appointments.  If it's bigger than 0
// Go for it!!

    if (apptCounts[0] > 0)  li = MemHandleLock(apptLists[0]);
    
    total = apptCounts[0]; // number of appts today
    totalapt = total;
    
    display = 0;
    
    if (total > 5) total = 5;
    
    for (i=0; display < total; i++) {

     offset = 83+(FntCharHeight()*j);
     
     recHandle = DmQueryRecord(DatebookDB,
            li[i].recordNum);
     apptrecP = MemHandleLock(recHandle);

        // If the appointment is untimed OR time of event > current time
        if ((TimeToInt(apptrecP->when.startTime) == -1) ||
           ((apptrecP->when.startTime.hours*100+apptrecP->when.startTime.minutes) > (Now.hour*100+Now.minute)))
           {
           	// Don't convert time to string if it's untimed
           	if (TimeToInt(apptrecP->when.startTime) != -1) {
              TimeToAscii(apptrecP->when.startTime.hours,
                          apptrecP->when.startTime.minutes, 2, AppErrStr);
             WinDrawChars( AppErrStr, StrLen( AppErrStr ), 1, offset);
           	}
           	else // untimed event
           	{
             untimed = untimed + 1; // Increment the number of untimed events
             if (untimed <= 2) // Don't display anything for the time if we have more than 2 untimed events
               {
                 WinDrawChars( "N/A", 3, 1, offset);
               }  
           	}
           	
            /* Calculate the position of the description */
            l1 = sizeof(ApptDateTimeType)+sizeof(ApptDBRecordFlags);
            if (apptrecP->flags.alarm) l1 += sizeof(AlarmInfoType);
            if (apptrecP->flags.repeat) l1 += sizeof(RepeatInfoType);
            if (apptrecP->flags.exceptions)
                l1 += sizeof(UInt) +
                    (((ExceptionsListType *)
                    ((CharPtr) apptrecP+l1))->numExceptions*sizeof(DateType));

             // Print description if the flag is set
             if (apptrecP->flags.description)
             {
               if ((TimeToInt(apptrecP->when.startTime) != -1) || 
                  ((TimeToInt(apptrecP->when.startTime) == -1) && 
                   (untimed <= 2))) 
               {
                  WinDrawChars((CharPtr) apptrecP+l1, StrLen((CharPtr) apptrecP+l1), 40, offset);
                  j = j + 1;
                  display = display + 1;
               }   
               
             }
             else
               {
               	WinDrawChars("N/A", 3, 40, offset);
                display = display + 1;
               }
            
        } // End if date > current time
        else // Skip over old appointment
        {
        	total = total + 1;
        	
        	if (total > totalapt) {
        	   total = totalapt;
        	}
        }

    MemHandleUnlock(recHandle);

    }


    for (i=0; i < 31; i++) {
        if (apptLists[i]) {
            MemHandleFree(apptLists[i]);
            apptLists[i] = 0;
            apptCounts[i] = 0;
        }
    }

    if (DatebookDB) DmCloseDatabase(DatebookDB);

   }


		
}

/***********************************************************************
 *
 * FUNCTION:    PilotMain
 *
 * DESCRIPTION: This is the main entry point for the application.
 *
 * PARAMETERS:  cmd - word value specifying the launch code. 
 *              cmdPB - pointer to a structure that is associated with the launch code. 
 *              launchFlags -  word value providing extra information about the launch.
 * RETURNED:    Result of launch
 *
 * REVISION HISTORY: 
 *
 *
 ***********************************************************************/
UInt32 PilotMain(UInt16 cmd, MemPtr cmdPBP, UInt16 launchFlags)
{
	Err error = errNone;
    
	switch (cmd) {
		case sysAppLaunchCmdNormalLaunch:
		    
			if ((error = AppStart()) == 0) {			
    		    FossilSetWatchApp( true );
				FossilDisplayRefreshRateSet( kFossilRefreshAuto );
		        WinPushDrawState();
        		WinSetDrawWindow( WinGetDisplayWindow() );
				draw_watch();
				draw_battery();
				play_alarm();
				AppEventLoop();
				AppStop();
				WinPopDrawState();
     			FossilDisplayRefreshRateSet( kFossilRefreshDefault );
     			FossilSetWatchApp( false );
			}
			break;

		case wpdaAppLaunchWatchDrawTime:
          WinPushDrawState();
    	  WinSetDrawWindow( WinGetDisplayWindow() );
		  draw_watch();
		  draw_battery();
		  handleAlarm();
		  WinPopDrawState();
		  EvtSetAutoOffTimer(SetExactly,2);
        break;
        
		case wpdaAppLaunchWatchFaceNext:
		  WinPushDrawState();
          WinSetDrawWindow( WinGetDisplayWindow() );
		  draw_watch();
		  draw_battery();
		  handleAlarm();
		  WinPopDrawState();
        break;

		case wpdaAppLaunchWatchFacePrev:
		  WinPushDrawState();
          WinSetDrawWindow( WinGetDisplayWindow() );
		  draw_watch();
		  draw_battery();
		  handleAlarm();
		  WinPopDrawState();
        break;

		default:
			break;
	}
	
	return error;
}


