// Enigma's Game Editor
// Copyright (C) 2001 Brian Enigma <enigma@netninja.com>
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version. 
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details. 
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
//
// $Id: general.c,v 1.8 2001/06/26 09:06:12 enigma Exp $

#include <PalmOS.h>

UInt32 min(UInt32 a, UInt32 b)
{
    if (a < b)
        return a;
    return b;
}

static FieldPtr SetFieldTextFromHandle(UInt16 fieldID, MemHandle textHandle)
{
    MemHandle oldTextHandle;
    FormPtr form = FrmGetActiveForm();
    FieldPtr fieldPtr;

    // Get the field and the field's current text handle
    fieldPtr = FrmGetObjectPtr(form, FrmGetObjectIndex(form, fieldID));
    ErrNonFatalDisplayIf(!fieldPtr, "missing field");
    oldTextHandle = FldGetTextHandle(fieldPtr);
    
    // Set the field's text to the new text
    FldSetTextHandle(fieldPtr, textHandle);
    FldDrawField(fieldPtr);

    // Free the handle AFTER we call FldSetTextHandle()
    if (oldTextHandle)
        MemHandleFree(oldTextHandle);
    return fieldPtr;
}

static FieldPtr SetFieldTextFromStr(UInt16 fieldID, char *stringPtr)
{
    MemHandle textHandle;

    // Get some space in which to stash the string
    textHandle = MemHandleNew(StrLen(stringPtr) + 1);
    if (!textHandle)
        return NULL;

    // Copy the string to the locked handle
    StrCopy(MemHandleLock(textHandle), stringPtr);

    // Unlock the string handle
    MemHandleUnlock(textHandle);

    // Set the field to the handle
    return SetFieldTextFromHandle(fieldID, textHandle);
}

static void ClearFieldText(UInt16 fieldID)
{
    SetFieldTextFromHandle(fieldID, NULL);
}

static MemHandle GetFieldHandle(UInt16 fieldID)
{
    MemHandle oldTextHandle;
    FormPtr form = FrmGetActiveForm();
    FieldPtr fieldPtr;

    // Get the field and the field's current text handle
    fieldPtr = FrmGetObjectPtr(form, FrmGetObjectIndex(form, fieldID));
    ErrNonFatalDisplayIf(!fieldPtr, "missing field");
    oldTextHandle = FldGetTextHandle(fieldPtr);
    return oldTextHandle;
}

#define TYPE_BYTE  0
#define TYPE_INT   1
#define TYPE_LONG  2
#define TYPE_INT12 3
static void saveFieldToFile(UInt16 fieldId, 
                            int dataType, 
                            LocalID dbId, 
                            UInt32 recordId, 
                            UInt32 offset)
{
    DmOpenRef db = 0;
    MemHandle rec;
    MemHandle textField;
    MemPtr p;
    MemPtr s;
    FormPtr form = FrmGetActiveForm();
    unsigned char bValue;
    UInt16 iValue;
    UInt32 lValue;
    
    db = DmOpenDatabase(0, dbId, dmModeReadWrite);
    if (db)
    {
        rec = DmGetRecord(db, recordId);
        p = MemHandleLock(rec);
        textField = GetFieldHandle(fieldId);
        s = MemHandleLock(textField);
        lValue = StrAToI(s);
        switch (dataType)
        {
            case TYPE_BYTE:
                bValue = min(lValue, 255);
                break;
            case TYPE_INT:
                iValue = min(lValue, 65535);
                break;
            case TYPE_INT12:
                iValue = min(lValue, 4095);
                break;
            case TYPE_LONG:
                lValue = lValue;
                break;
        }
        MemHandleUnlock(textField);
        switch (dataType)
        {
            case TYPE_BYTE:
                DmWrite(p, offset, &bValue, 1);
                break;
            case TYPE_INT:
            case TYPE_INT12:
                DmWrite(p, offset, &iValue, 2);
                break;
            case TYPE_LONG:
                DmWrite(p, offset, &lValue, 4);
                break;
        }
        MemHandleUnlock(rec);
        DmReleaseRecord(db, recordId, true);
        DmCloseDatabase(db);
    }
}

static void saveFieldToFileByName(  UInt16 fieldId, 
                                    int dataType, 
                                    char *fileName, 
                                    UInt32 recordId, 
                                    UInt32 offset)
{
    LocalID dbId = DmFindDatabase(0, fileName);
    if (dbId)
            saveFieldToFile(fieldId, dataType, dbId, recordId, offset);
}

static Boolean loadFieldFromFile(  UInt16 fieldId, 
                                int dataType, 
                                LocalID dbId, 
                                UInt32 recordId, 
                                UInt32 offset)
{
    Boolean result = false;
    DmOpenRef db = 0;
    MemHandle rec;
    MemPtr p;
    unsigned char bytes = 0;
    UInt32 lValue;
    char buff[20];
    unsigned char i;

    /*
    if ( (dataType != TYPE_BYTE) && ((offset & 2) == 1))
    {
        offset--;
        smallOffset += 8;
    }
    */
    db = DmOpenDatabase(0, dbId, dmModeReadWrite);
    if (db)
    {
        rec = DmGetRecord(db, recordId);
        if (rec)
        {
            p = MemHandleLock(rec);
            switch (dataType)
            {
                case TYPE_BYTE:
                    bytes = 1;
                    break;
                case TYPE_INT:
                case TYPE_INT12:
                    bytes = 2;
                    break;
                case TYPE_LONG:
                    bytes = 4;
                    break;
            }
            lValue = 0;
            for (i=0; i<bytes; i++)
            {
                lValue *= 0x100;
                lValue += *((unsigned char *)(p+offset+i));
            }
            StrIToA(buff, lValue);
            SetFieldTextFromStr(fieldId, buff);
            MemHandleUnlock(rec);
            DmReleaseRecord(db, recordId, false);
            DmCloseDatabase(db);
            result = true;
        }
    }
    return result;
}

static Boolean loadFieldFromFileByName(UInt16 fieldId, 
                                    int dataType, 
                                    char *fileName, 
                                    UInt32 recordId, 
                                    UInt32 offset)
{
    LocalID dbId = DmFindDatabase(0, fileName);
    if (dbId)
    {
        return loadFieldFromFile(fieldId, dataType, dbId, recordId, offset);
    }
    return false;
}

static Boolean loadFieldFromFileByNameAndRecName(   UInt16 fieldId,
                                                    int dataType,
                                                    char *fileName,
                                                     UInt32 recordType,
                                                     UInt32 offset)
{
    Boolean result = false;
    int i=0, max=0;
    DmOpenRef db = 0;
    UInt32 recordId = 0;
    MemHandle rec;
    UInt32 resType;
    MemPtr p;
    unsigned char *bValue;
    UInt16 *iValue;
    UInt32 *lValue;
    char buff[20];
    LocalID dbId = DmFindDatabase(0, fileName);
    if (dbId)
        db = DmOpenDatabase(0, dbId, dmModeReadWrite);
    if (db)
        recordId = DmFindResourceType(db, recordType, 0);
    if (recordId != -1)
    {
        rec = DmGetResourceIndex(db, recordId);
        if (rec)
        {
            //SndPlaySystemSound(sndError);
            p = MemHandleLock(rec);
            switch (dataType)
            {
                case TYPE_BYTE:
                    bValue = (unsigned char *)(p + offset);
                    StrIToA(buff, *bValue);
                    break;
                case TYPE_INT:
                case TYPE_INT12:
                    iValue = (UInt16 *)(p + offset);
                    StrIToA(buff, *iValue);
                    break;
                case TYPE_LONG:
                    lValue = (UInt32 *)(p + offset);
                    StrIToA(buff, *lValue);
                    break;
            }
            SetFieldTextFromStr(fieldId, buff);
            MemHandleUnlock(rec);
            result = true;
        }
    }
    if (db)
        DmCloseDatabase(db);
    return result;
}

static void saveFieldToFileByNameAndRecName(  UInt16 fieldId,
                                                int dataType,
                                                char *fileName,
                                                UInt32 recordType,
                                                UInt32 offset)
{
    int i=0, max=0;
    DmOpenRef db = 0;
    UInt32 recordId = 0;
    MemHandle rec;
    UInt32 resType;
    unsigned char bValue;
    UInt16 iValue;
    UInt32 lValue;
    MemHandle textField;
    MemPtr p;
    MemPtr s;
    LocalID dbId = DmFindDatabase(0, fileName);
    if (dbId)
        db = DmOpenDatabase(0, dbId, dmModeReadWrite);
    if (db)
        recordId = DmFindResourceType(db, recordType, 0);
    if (recordId != -1)
    {
        rec = DmGetResourceIndex(db, recordId);
        if (rec)
        {
            p = MemHandleLock(rec);
            textField = GetFieldHandle(fieldId);
            s = MemHandleLock(textField);
            lValue = StrAToI(s);
            switch (dataType)
            {
                case TYPE_BYTE:
                    bValue = min(lValue, 255);
                    break;
                case TYPE_INT:
                    iValue = min(lValue, 65535);
                    break;
                case TYPE_INT12:
                    iValue = min(lValue, 4095);
                    break;
                case TYPE_LONG:
                    lValue = lValue;
                    break;
            }
            MemHandleUnlock(textField);
            switch (dataType)
            {
                case TYPE_BYTE:
                    DmWrite(p, offset, &bValue, 1);
                    break;
                case TYPE_INT:
                case TYPE_INT12:
                    DmWrite(p, offset, &iValue, 2);
                    break;
                case TYPE_LONG:
                    DmWrite(p, offset, &lValue, 4);
                    break;
            }
            MemHandleUnlock(rec);
        }
    }
    if (db)
        DmCloseDatabase(db);
}
